<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class AiScanDatabase extends Command
{
    protected $signature = 'ai:scan-db {--with-counts : Include row counts per table}';
    protected $description = 'Generate a safe schema summary for the AI assistant (tables, columns, optional counts).';

    public function handle(): int
    {
        $dbName = config('database.connections.' . config('database.default') . '.database');
        if (!$dbName) {
            $this->error('Unable to detect DB name.');
            return 1;
        }

        $this->info('Scanning schema for database: ' . $dbName);

        $columns = DB::table('information_schema.columns')
            ->select('table_name', 'column_name', 'data_type')
            ->where('table_schema', $dbName)
            ->orderBy('table_name')
            ->orderBy('ordinal_position')
            ->get();

        $byTable = [];
        foreach ($columns as $col) {
            $t = (string) $col->table_name;
            if (!isset($byTable[$t])) $byTable[$t] = ['columns' => [], 'count' => null];
            $byTable[$t]['columns'][] = [
                'name' => (string) $col->column_name,
                'type' => (string) $col->data_type,
            ];
        }

        if ($this->option('with-counts')) {
            foreach (array_keys($byTable) as $table) {
                try {
                    $cnt = (int) DB::table($table)->count();
                    $byTable[$table]['count'] = $cnt;
                } catch (\Throwable $e) {
                    $byTable[$table]['count'] = null;
                }
            }
        }

        $payload = [
            'database' => $dbName,
            'generated_at' => now()->toIso8601String(),
            'tables' => $byTable,
        ];

        $path = 'ai/schema.json';
        Storage::disk('local')->put($path, json_encode($payload, JSON_PRETTY_PRINT));
        $this->info('Schema summary written to storage/app/' . $path);
        return 0;
    }
}

